<?php
ini_set('display_errors', '1');
ini_set('display_startup_errors', '1');
error_reporting(E_ALL);

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/vendor/autoload.php';

use Mpdf\Mpdf;
use Mpdf\Config\ConfigVariables;
use Mpdf\Config\FontVariables;
use Endroid\QrCode\Builder\Builder;
use Endroid\QrCode\Writer\PngWriter;
use Endroid\QrCode\Encoding\Encoding;
use Endroid\QrCode\Color\Color;

/* ======================= إعدادات قابلة للتعديل ======================= */
/* حجم الصفحة (مم) */
const PAGE_W_MM = 77.99;
const PAGE_H_MM = 72.99;

/* ملف الخلفية (يجب أن يكون بجوار هذا الملف) */
const BACKGROUND_FILE = 'background2.png';

/* -------- QR CONTROL -------- */
const QR_W_MM       = 25;     // عرض QR (مم)
const QR_H_MM       = 25;     // ارتفاع QR (مم)
const QR_ANCHOR_X   = 'RIGHT';// 'LEFT' أو 'RIGHT'
const QR_OFFSET_X_MM= 40.2;     // إزاحة أفقية من المرسى (مم)
const QR_ANCHOR_Y   = 'TOP';  // 'TOP' أو 'BOTTOM'
const QR_OFFSET_Y_MM= 29;     // إزاحة عمودية من المرسى (مم)

/* -------- BARCODE BLOCK CONTROL --------
   هذا هو “البلوك” الحاوي للباركود + رقم الاستيكر.
   تقدر تضبط عرضه/ارتفاعه وإزاحته ومرساته وخلفيته وحدوده وحشوه الداخلي.
*/
const BC_BLOCK_W_MM   = 20;      // عرض البلوك (مم)
const BC_BLOCK_H_MM   = 2;      // ارتفاع البلوك (مم) — ثابت ومباشر
const BC_ANCHOR_X     = 'RIGHT'; // 'LEFT' أو 'RIGHT'
const BC_OFFSET_X_MM  = 13;       // إزاحة أفقية من المرسى (مم)
const BC_ANCHOR_Y     = 'TOP';   // 'TOP' أو 'BOTTOM'
const BC_OFFSET_Y_MM  = 55.5;      // إزاحة عمودية من المرسى (مم)
const BC_BG_DRAW      = true;    // ارسم خلفية للبلوك؟
const BC_BG_PADDING_MM= 1.0;     // حشوة حول المحتوى داخل البلوك (مم)
const BC_BG_COLOR     = [255,255,255]; // خلفية البلوك (RGB)
const BC_BORDER_MM    = 0.0;     // سمك الإطار (مم) - 0 لإلغاء
const BC_BORDER_COLOR = [0,0,0]; // لون الإطار (RGB)

/* -------- BARCODE CONTROL --------
   عرض الباركود يتحكم به BC_SIZE (أصغر = أضيق، أكبر = أعرض).
   الارتفاع بالمليمتر عبر BC_HEIGHT_MM.
   المحاذاة داخل البلوك عبر BC_ALIGN: 'center' أو 'left' أو 'right'.
*/
const BC_TYPE       = 'C128B';
const BC_SIZE       = 0.39;   // 0.18 ~ 0.40 (قلّل لو صار عريض/أسود)
const BC_HEIGHT_MM  = 2;     // ارتفاع الباركود (مم)
const BC_ALIGN      = 'center';// 'left' | 'center' | 'right'

/* -------- STICKER TEXT CONTROL -------- */
const STICKER_SHOW_TEXT     = true;
const STICKER_FONT_MM       = 1.8;  // حجم خط الرقم (مم تقريبي)
const STICKER_BOLD          = true; // خط عريض؟
const STICKER_LETTER_SP_MM  = 0;  // تباعد الحروف (مم)
const STICKER_MARGIN_TOP_MM = -0.2;  // مسافة أسفل الباركود (مم)
const STICKER_ALIGN         = 'center'; // 'left' | 'center' | 'right'
/* ===================================================================== */

/* اتصال DB */
$pdo = db();
if (!($pdo instanceof PDO)) { http_response_code(500); exit('DB error'); }

/* id */
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
  exit("<h2 style='color:red; text-align:center;'>⚠️ رقم الشهادة غير صالح.</h2>");
}
$id = (int) $_GET['id'];

/* السجل */
$stmt = $pdo->prepare("SELECT * FROM inspections WHERE id = ? LIMIT 1");
$stmt->execute([$id]);
$cert = $stmt->fetch(PDO::FETCH_ASSOC);
if (!$cert) { exit("<h2 style='color:red; text-align:center;'>⚠️ السجل غير موجود.</h2>"); }

/* مساعد */
if (!function_exists('h')) {
  function h($v){ return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8'); }
}
$sticker = isset($cert['stickerNum']) ? trim((string)$cert['stickerNum']) : '';

/* mPDF بخط Cairo فقط */
$defaultConfig     = (new ConfigVariables())->getDefaults();
$fontDirs          = $defaultConfig['fontDir'];
$defaultFontConfig = (new FontVariables())->getDefaults();
$fontData = array_replace_recursive($defaultFontConfig['fontdata'], [
  'cairo' => [
    'R' => 'Cairo-Regular.ttf',
    'B' => 'Cairo-Bold.ttf', // العريض المستخدَم
    'useOTL' => 0xFF,
    'useKashida' => 75,
  ],
]);

$mpdf = new Mpdf([
  'mode'         => 'utf-8',
  'format'       => [PAGE_W_MM, PAGE_H_MM],
  'orientation'  => 'P',
  'margin_left'  => 0,
  'margin_right' => 0,
  'margin_top'   => 0,
  'margin_bottom'=> 0,
  'fontDir'      => array_merge($fontDirs, [__DIR__ . '/fonts']),
  'fontdata'     => $fontData,
  'default_font' => 'cairo',
]);
$mpdf->autoScriptToLang = true;
$mpdf->autoLangToFont   = false;
$mpdf->SetDirectionality('rtl');
$mpdf->SetFont('cairo', '', 0, true);
$mpdf->SetAutoPageBreak(false);
$mpdf->SetDisplayMode('fullpage');

/* صفحة بالحجم المطلوب */
$mpdf->AddPageByArray([
  'orientation' => 'P',
  'sheet-size'  => [PAGE_W_MM, PAGE_H_MM],
  'margin-left' => 0, 'margin-right' => 0, 'margin-top' => 0, 'margin-bottom' => 0,
]);

/* 1) الخلفية أولًا (خلف كل العناصر) */
$bgAbs = realpath(__DIR__ . '/' . BACKGROUND_FILE);
if ($bgAbs && is_file($bgAbs)) {
  $mpdf->Image($bgAbs, 0, 0, PAGE_W_MM, PAGE_H_MM, '', '', true, false);
}


/* اجلب رقم الاستيكر/الـ SN من السجل */
$sn = '';
if (isset($cert['sn']) && $cert['sn'] !== '') {
  $sn = trim((string)$cert['sn']);
} else {
  $sn = isset($cert['stickerNum']) ? trim((string)$cert['stickerNum']) : '';
}

if ($sn === '') {
  exit("<h3 style='color:red; text-align:center'>لا يوجد SN (stickerNum) في السجل #{$id}.</h3>");
}

/* لو عندك دومين عام لتفادي localhost على الجوال */
$PUBLIC_BASE = ''; // مثال: 'https://pti.sseso.site' — اتركه فارغًا إن لا يوجد

/* ابنِ BASE يدعم البروكسي */
$xfProto = $_SERVER['HTTP_X_FORWARDED_PROTO'] ?? '';
if ($xfProto !== '') {
  $scheme = strtolower(trim(explode(',', $xfProto)[0]));
  if ($scheme !== 'https' && $scheme !== 'http') $scheme = 'https';
} else {
  $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') ? 'https' : 'http';
}

$xfHost = $_SERVER['HTTP_X_FORWARDED_HOST'] ?? '';
if ($xfHost !== '') {
  $host = trim(explode(',', $xfHost)[0]);
} else {
  $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
}

$base = $scheme . '://' . $host;

/* بدّل إلى PUBLIC_BASE لو كنا على بيئة محلية/شبكة داخلية */
if ($PUBLIC_BASE !== '' && preg_match('~^https?://(localhost|127\.0\.0\.1|10\.\d+\.\d+\.\d+|192\.168\.\d+\.\d+|172\.(1[6-9]|2\d|3[0-1])\.\d+\.\d+)(:\d+)?$~i', $base)) {
  $base = rtrim($PUBLIC_BASE, '/');
}

/* صفحة الاستعلام والباراميتر */
$verifyPath = '/apt/inspection_status'; // ← بدل index.html
$queryParam = 'sn';

/* رابط الاستعلام النهائي */
$qrLink = rtrim($base, '/') . $verifyPath . '?' . $queryParam . '=' . rawurlencode($sn);


/* أنشئ صورة الـ QR (Endroid v4) */
$qr = \Endroid\QrCode\Builder\Builder::create()
  ->writer(new \Endroid\QrCode\Writer\PngWriter())
  ->data($qrLink)
  ->encoding(new \Endroid\QrCode\Encoding\Encoding('UTF-8'))
  ->size(800)
  ->margin(0)
  ->foregroundColor(new \Endroid\QrCode\Color\Color(0,0,0))
  ->backgroundColor(new \Endroid\QrCode\Color\Color(255,255,255))
  ->build();

$qrTmp = __DIR__ . '/qr_tmp_' . $id . '.png';
file_put_contents($qrTmp, $qr->getString());


/* احسب موضع QR بحسب المرسى */
$qrX = (strtoupper(QR_ANCHOR_X) === 'LEFT')
      ? QR_OFFSET_X_MM
      : (PAGE_W_MM - QR_OFFSET_X_MM - QR_W_MM);
$qrY = (strtoupper(QR_ANCHOR_Y) === 'TOP')
      ? QR_OFFSET_Y_MM
      : (PAGE_H_MM - QR_OFFSET_Y_MM - QR_H_MM);

/* ارسم الـQR فوق الخلفية */
$mpdf->Image($qrTmp, $qrX, $qrY, QR_W_MM, QR_H_MM, 'png', '', true, false);

/* 3) بلوك الباركود + رقم الاستيكر */
if ($sticker !== '') {

  /* موضع البلوك حسب المرسى */
  $bcX = (strtoupper(BC_ANCHOR_X) === 'LEFT')
        ? BC_OFFSET_X_MM
        : (PAGE_W_MM - BC_OFFSET_X_MM - BC_BLOCK_W_MM);
  $bcY = (strtoupper(BC_ANCHOR_Y) === 'TOP')
        ? BC_OFFSET_Y_MM
        : (PAGE_H_MM - BC_OFFSET_Y_MM - BC_BLOCK_H_MM);

  /* خلفية + إطار للبلوك حسب الرغبة */
  if (BC_BG_DRAW) {
    [$r,$g,$b] = BC_BG_COLOR;
    $mpdf->SetFillColor($r,$g,$b);
    if (BC_BORDER_MM > 0) {
      [$br,$bg,$bb] = BC_BORDER_COLOR;
      $mpdf->SetDrawColor($br,$bg,$bb);
      $mpdf->SetLineWidth(BC_BORDER_MM);
      $mpdf->Rect($bcX, $bcY, BC_BLOCK_W_MM, BC_BLOCK_H_MM, 'DF'); // Fill + Stroke
    } else {
      $mpdf->Rect($bcX, $bcY, BC_BLOCK_W_MM, BC_BLOCK_H_MM, 'F');  // Fill only
    }
  }

  /* محتوى البلوك (باركود + رقم) */
  $taBC = (BC_ALIGN === 'left' ? 'left' : (BC_ALIGN === 'right' ? 'right' : 'center'));
  $taTX = (STICKER_ALIGN === 'left' ? 'left' : (STICKER_ALIGN === 'right' ? 'right' : 'center'));

  $innerW = BC_BLOCK_W_MM - (BC_BG_PADDING_MM * 2);
  $innerH = BC_BLOCK_H_MM - (BC_BG_PADDING_MM * 2);

  $bcHtml  = '<div style="width:'.$innerW.'mm;height:'.$innerH.'mm;';
  $bcHtml .= 'box-sizing:border-box;';
  $bcHtml .= 'padding:0;';
  $bcHtml .= 'text-align:'.$taBC.';';
  $bcHtml .= 'line-height:1.15;">';

  // باركود
  $bcHtml .= '<div>'
          .  '<barcode code="'.h($sticker).'" type="'.BC_TYPE.'" '
          .  'size="'.BC_SIZE.'" height="'.BC_HEIGHT_MM.'" text="0" padding="0" />'
          .  '</div>';

  // نص الاستيكر
  if (STICKER_SHOW_TEXT) {
    $fw = STICKER_BOLD ? 'bold' : 'normal';
    $bcHtml .= '<div style="margin-top:'.STICKER_MARGIN_TOP_MM.'mm;'
         .  'font-size:'.STICKER_FONT_MM.'mm;'
         .  'font-weight:'.$fw.';'
         .  'letter-spacing:'.STICKER_LETTER_SP_MM.'mm;'
         .  'text-align:'.$taTX.';'
         .  'white-space:nowrap;'
         .  'line-height:1;'
         .  'direction:ltr; unicode-bidi:embed;">'
         .  h($sticker).'</div>';
  }

  $bcHtml .= '</div>';

  // نكتب المحتوى داخل البلوك بإحداثيات ثابتة مع حشوة داخلية
  $mpdf->WriteFixedPosHTML(
    $bcHtml,
    $bcX + BC_BG_PADDING_MM,    // X داخل البلوك
    $bcY + BC_BG_PADDING_MM,    // Y داخل البلوك
    $innerW,                    // عرض المساحة الداخلية
    $innerH,                    // ارتفاع المساحة الداخلية
    'auto'
  );
}

/* مهم: امنع أي مخرجات مفاجئة وقت الإخراج */
$prevDisplay = ini_get('display_errors');
ini_set('display_errors', '0');

/* نظّف أي buffers مفتوحة (لو فيه مسافات/تحذيرات انطبعت) */
while (ob_get_level() > 0) { ob_end_clean(); }

/* ===== 1) توليد اسم الملف: (رقم اللوحة) - (نوع الوثيقة).pdf ===== */

// رقم اللوحة (ندعم عمودين محتملين plateNum/plate_number + تطبيع الأرقام العربية)
$plate_number_raw = trim($cert['plateNum'] ?? $cert['plate_number'] ?? '');
if ($plate_number_raw === '') {
    $plate_number_raw = 'بدون-لوحة';
}
$plate_number = strtr($plate_number_raw, [
    '٠'=>'0','١'=>'1','٢'=>'2','٣'=>'3','٤'=>'4',
    '٥'=>'5','٦'=>'6','٧'=>'7','٨'=>'8','٩'=>'9'
]);
$plate_number = preg_replace('/\s+/u', ' ', $plate_number);
$plate_number = trim($plate_number);

// نوع الوثيقة (خذ من قاعدة البيانات إن وُجد وإلا استخدم نصًا ثابتًا)
$doc_type = trim('ملصق الفحص الدوري');

// تركيب الاسم وتنظيفه من محارف غير صالحة للأسماء
$proposed_name = $plate_number . ' - ' . $doc_type;
$clean_name = preg_replace('/[\\\\\/:*?"<>|]/u', '_', $proposed_name);
$clean_name = preg_replace('/\s+/u', ' ', $clean_name);
$clean_name = preg_replace('/_+/u', '_', $clean_name);
$clean_name = trim($clean_name, " \t\n\r\0\x0B-_");

// اسم الملف النهائي
$custom_filename = $clean_name . '.pdf';

// مجلد الحفظ temp/
$dir = __DIR__ . '/temp';
if (!is_dir($dir)) {
    mkdir($dir, 0777, true);
}
$save_path = $dir . '/' . $custom_filename;

/* ===== 2) حفظ الـ PDF على القرص ===== */
$mpdf->Output($save_path, \Mpdf\Output\Destination::FILE);

/* ===== 3) بناء رابط عام وفتح نافذة الطباعة داخل IFRAME ===== */
$protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
$host     = $_SERVER['HTTP_HOST'];
$basePath = rtrim(dirname($_SERVER['PHP_SELF']), '/');
$public_url = $protocol . '://' . $host . $basePath . '/temp/' . rawurlencode($custom_filename);

// أعِد تفعيل عرض الأخطاء لو حاب ترجع الإعداد كما كان
ini_set('display_errors', $prevDisplay);

/* ===== 4) صفحة عرض/طباعة الملف المحفوظ ===== */
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="UTF-8">
  <title><?= htmlspecialchars($doc_type, ENT_QUOTES, 'UTF-8') ?></title>
  <style>
    html, body { margin: 0; height: 100%; overflow: hidden; background:#f7f7f7; }
    iframe { width: 100%; height: 100%; border: none; }
  </style>
</head>
<body>
  <iframe id="pdfFrame"
          src="<?= htmlspecialchars($public_url, ENT_QUOTES, 'UTF-8') . '?v=' . time() ?>">
  </iframe>
  <script>
    const f = document.getElementById('pdfFrame');
    // عند تحميل الـPDF داخل الإطار، نفّذ الطباعة
    f.addEventListener('load', () => {
      setTimeout(() => {
        try { f.contentWindow?.focus(); f.contentWindow?.print(); } catch(e) {}
      }, 800);
    });
  </script>
</body>
</html>
<?php
exit;