<?php
/**
 * File: edit_inspection.php
 * نموذج تعديل جدول inspections — يشمل كل الأعمدة في الـDDL
 */

require_once __DIR__ . '/config.php';

// اتصال PDO
if (!isset($pdo)) {
  if (!isset($dsn, $db_user, $db_pass)) die('DB config missing');
  $pdo = new PDO($dsn, $db_user, $db_pass, [
    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    PDO::MYSQL_ATTR_INIT_COMMAND => 'SET NAMES utf8mb4'
  ]);
}

$id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
if ($id <= 0) die('معرّف غير صالح');

// جلب السجل
$stm = $pdo->prepare('SELECT * FROM inspections WHERE id = ? LIMIT 1');
$stm->execute([$id]);
$rec = $stm->fetch();
if (!$rec) die('لم يتم العثور على السجل');

// تعريف الأعمدة
$cols = $pdo->query('DESCRIBE inspections')->fetchAll();
$meta = [];
foreach ($cols as $c) { $meta[$c['Field']] = $c; }

// المدن
$cities = [];
try {
  $cities = $pdo->query('SELECT DISTINCT city_name_ar FROM cities ORDER BY city_name_ar')->fetchAll(PDO::FETCH_COLUMN);
} catch (Throwable $e) {
  $cities = ['الرياض','جدة','مكة','المدينة المنورة','الدمام','الخبر','الظهران','الطائف','تبوك','بريدة','أبها','خميس مشيط','جازان','حائل','ينبع','الجبيل','نجران','القصيم'];
}

// أنواع التقارير
$reportTypesFromDB = [];
try {
  $reportTypesFromDB = $pdo->query("SELECT DISTINCT report_type FROM inspections WHERE report_type IS NOT NULL AND report_type <> '' ORDER BY report_type")
                           ->fetchAll(PDO::FETCH_COLUMN);
} catch (Throwable $e) { $reportTypesFromDB = []; }

$defaults = ['الفحص الفني الدوري','فحص باصات الحج'];
$reportTypes = array_unique(array_merge($defaults, $reportTypesFromDB));

function h($v){ return htmlspecialchars((string)$v, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); }
function toDate($v){ if (!$v) return ''; $t=strtotime($v); return $t?date('Y-m-d',$t):''; }
function toDateTimeLocal($v){ if (!$v) return ''; $t=strtotime($v); return $t?date('Y-m-d\TH:i',$t):''; }

// ترتيب الحقول حسب DDL (بدون newInspLoc)
$fields = [
  'report_type','ownerName','plateNum','chassisNum','manufacturer','modelYear','model','color','odometer','serialNum','docSerialNum',
  'inspDateGregorian','inspDateHijri','inspectionIssueDate','inspectionExpiryDate','validUntilDocGregorian','validUntilDocHijri',
  'inspAuthDoc','inspLoc','laneNumDoc','inspNumDoc','stickerNum','attemptNumDoc',
  'sideSlipTest','sideSlipPass','frontBrakeEff','frontBrakePass','frontBrakeImb','frontBrakeImbPass','frontBrakeFric','frontBrakeFricPass',
  'rearBrakeEff','rearBrakePass','rearBrakeImb','rearBrakeFric','rearBrakeFricPass','handBrakeEff','handBrakePass',
  'rightLightInt','rightLightPass','rightLightDevVert','rightLightDevVertPass','rightLightDevHoriz','rightLightDevHorizPass',
  'leftLightInt','leftLightPass','leftLightDevVert','leftLightDevVertPass','leftLightDevHoriz','leftLightDevHorizPass',
  'coLevel','coPass','smokeDensity','smokePass','hcLevel','hcPass','extInsp','brakeTest','underbodyTest','status',
  'created_at', // عرض فقط
  'sn'          // عرض فقط (Virtual)
];

// تسميات عربية
$labels = [
  'report_type'=>'نوع التقرير','ownerName'=>'اسم المالك','plateNum'=>'رقم اللوحة','chassisNum'=>'رقم الهيكل','manufacturer'=>'الشركة المصنّعة','modelYear'=>'سنة الموديل','model'=>'الطراز','color'=>'اللون','odometer'=>'قراءة العداد','serialNum'=>'الرقم التسلسلي','docSerialNum'=>'رقم الوثيقة',
  'inspDateGregorian'=>'تاريخ الفحص (ميلادي)','inspDateHijri'=>'تاريخ الفحص (هجري)','inspectionIssueDate'=>'تاريخ إصدار الفحص','inspectionExpiryDate'=>'تاريخ انتهاء الفحص','validUntilDocGregorian'=>'صلاحية الوثيقة (ميلادي)','validUntilDocHijri'=>'صلاحية الوثيقة (هجري)',
  'inspAuthDoc'=>'جهة الفحص','inspLoc'=>'المدينة','laneNumDoc'=>'رقم المسار','inspNumDoc'=>'رقم الفحص','stickerNum'=>'رقم الملصق','attemptNumDoc'=>'عدد المحاولات',
  'sideSlipTest'=>'انحراف العجلات','sideSlipPass'=>'نجاح الانحراف','frontBrakeEff'=>'كفاءة الفرامل الأمامية','frontBrakePass'=>'نجاح الفرامل الأمامية','frontBrakeImb'=>'عدم اتزان أمامي','frontBrakeImbPass'=>'نجاح عدم الاتزان الأمامي','frontBrakeFric'=>'احتكاك أمامي','frontBrakeFricPass'=>'نجاح الاحتكاك الأمامي',
  'rearBrakeEff'=>'كفاءة الفرامل الخلفية','rearBrakePass'=>'نجاح الفرامل الخلفية','rearBrakeImb'=>'عدم اتزان خلفي','rearBrakeFric'=>'احتكاك خلفي','rearBrakeFricPass'=>'نجاح الاحتكاك الخلفي','handBrakeEff'=>'كفاءة فرامل اليد','handBrakePass'=>'نجاح فرامل اليد',
  'rightLightInt'=>'شدة إضاءة يمين','rightLightPass'=>'نجاح إضاءة يمين','rightLightDevVert'=>'انحراف رأسي يمين','rightLightDevVertPass'=>'نجاح الانحراف الرأسي يمين','rightLightDevHoriz'=>'انحراف أفقي يمين','rightLightDevHorizPass'=>'نجاح الانحراف الأفقي يمين',
  'leftLightInt'=>'شدة إضاءة يسار','leftLightPass'=>'نجاح إضاءة يسار','leftLightDevVert'=>'انحراف رأسي يسار','leftLightDevVertPass'=>'نجاح الانحراف الرأسي يسار','leftLightDevHoriz'=>'انحراف أفقي يسار','leftLightDevHorizPass'=>'نجاح الانحراف الأفقي يسار',
  'coLevel'=>'CO','coPass'=>'نجاح CO','smokeDensity'=>'كثافة الدخان','smokePass'=>'نجاح الدخان','hcLevel'=>'HC','hcPass'=>'نجاح HC','extInsp'=>'فحص خارجي','brakeTest'=>'اختبار الفرامل','underbodyTest'=>'فحص أسفل المركبة','status'=>'الحالة','created_at'=>'أُنشئ في','sn'=>'SN (مُولّد)'
];

// نوع الإدخال لكل حقل
function inputType($name, $meta){
  $t = strtolower($meta['Type'] ?? '');
  if ($name === 'status') return 'status';
  if ($name === 'report_type') return 'report_type';
  if ($name === 'inspLoc') return 'city';
  if ($name === 'created_at' || $name === 'sn') return 'readonly';
  if (str_contains($t,'datetime')) return 'datetime';
  if (str_contains($t,'date')) return 'date';
  if (preg_match('/int|decimal|float|double|smallint|tinyint|bigint/i',$t)) return 'number';
  return 'text';
}
?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>تعديل فحص دوري</title>
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.rtl.min.css">
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
  <style>
    body{background:#f5f7fb;font-family:'Cairo',system-ui,sans-serif}
    .container-narrow{max-width:1080px}
    .card{border:0;box-shadow:0 10px 24px rgba(0,0,0,.08);border-radius:14px}
    .grid{display:grid;grid-template-columns:repeat(12,1fr);gap:12px}
    .g-12{grid-column:span 12}.g-6{grid-column:span 6}.g-4{grid-column:span 4}.g-3{grid-column:span 3}
    @media(max-width:768px){.g-6,.g-4,.g-3{grid-column:span 12}}
    .sticky-actions{position:sticky;bottom:0;background:#fff;border-top:1px solid #eee;padding:12px}
    .badge-id{font-size:.9rem}
  </style>
</head>
<body>
<div class="container container-narrow py-3">
  <div class="d-flex justify-content-between align-items-center mb-2">
    <h4 class="mb-0">تعديل بيانات الفحص الدوري</h4>
    <span class="badge bg-secondary badge-id">ID: <?= (int)$rec['id'] ?></span>
  </div>

  <div class="card">
    <div class="card-body">
      <form method="post" action="update_inspection.php" class="grid" enctype="application/x-www-form-urlencoded">
        <input type="hidden" name="id" value="<?= (int)$rec['id'] ?>">
<?php
foreach ($fields as $f) {
  if (!array_key_exists($f,$rec) && $f!=='sn' && $f!=='created_at') continue;
  $label = $labels[$f] ?? $f;
  $v = $rec[$f] ?? ($f==='sn' ? ($rec['stickerNum'] ?? '') : '');
  $type = inputType($f, $meta[$f] ?? []);
  $colClass = in_array($f,['notes','description']) ? 'g-12' : 'g-6';
  if (in_array($f,[
    'rightLightDevVert','rightLightDevVertPass','rightLightDevHoriz','rightLightDevHorizPass',
    'leftLightDevVert','leftLightDevVertPass','leftLightDevHoriz','leftLightDevHorizPass'
  ])) $colClass='g-3';
?>
        <div class="<?= $colClass ?>">
          <label class="form-label" for="f_<?= h($f) ?>"><?= h($label) ?></label>
<?php if ($type==='readonly'): ?>
          <input id="f_<?= h($f) ?>" class="form-control" type="text" value="<?= h($v) ?>" readonly>
<?php elseif ($type==='status'): ?>
          <select id="f_<?= h($f) ?>" class="form-select" name="<?= h($f) ?>">
            <option value="active"   <?= (strtolower((string)$v)==='active'?'selected':'') ?>>نشط</option>
            <option value="inactive" <?= (strtolower((string)$v)==='inactive'?'selected':'') ?>>متوقف</option>
          </select>
<?php elseif ($type==='report_type'): ?>
          <select id="f_<?= h($f) ?>" class="form-select" name="<?= h($f) ?>">
            <?php foreach($reportTypes as $rt): $sel = ((string)$v===(string)$rt)?'selected':''; ?>
              <option value="<?= h($rt) ?>" <?= $sel ?>><?= h($rt) ?></option>
            <?php endforeach; ?>
          </select>
<?php elseif ($type==='city'): ?>
          <select id="f_<?= h($f) ?>" class="form-select" name="<?= h($f) ?>">
            <option value="">— اختر المدينة —</option>
            <?php foreach($cities as $c): $sel = ((string)$v===(string)$c)?'selected':''; ?>
              <option value="<?= h($c) ?>" <?= $sel ?>><?= h($c) ?></option>
            <?php endforeach; ?>
          </select>
<?php elseif ($type==='date'): ?>
          <input id="f_<?= h($f) ?>" class="form-control" type="date" name="<?= h($f) ?>" value="<?= h(toDate($v)) ?>">
<?php elseif ($type==='datetime'): ?>
          <input id="f_<?= h($f) ?>" class="form-control" type="datetime-local" name="<?= h($f) ?>" value="<?= h(toDateTimeLocal($v)) ?>">
<?php elseif ($type==='number'): ?>
          <input id="f_<?= h($f) ?>" class="form-control" type="number" step="any" name="<?= h($f) ?>" value="<?= h($v) ?>">
<?php else: ?>
          <input id="f_<?= h($f) ?>" class="form-control" type="text" name="<?= h($f) ?>" value="<?= h($v) ?>">
<?php endif; ?>
        </div>
<?php } ?>
        <div class="g-12 sticky-actions mt-2 d-flex justify-content-between align-items-center">
          <a href="javascript:history.back()" class="btn btn-outline-secondary"><i class="fa fa-arrow-right"></i> العودة إلى الصفحة الرئيسية</a>
          <div class="d-flex gap-2">
            <button type="reset" class="btn btn-light">إعادة تعيين</button>
            <button type="submit" class="btn btn-primary"><i class="fa fa-save"></i> حفظ التعديلات</button>
          </div>
        </div>
      </form>
    </div>
  </div>
</div>

<!-- ===== منطق توليد docSerialNum من التاريخ ===== -->
<script>
(function(){
  // عناصر الحقول
  const idVal = String(<?= (int)$rec['id'] ?>);
  const fDate = document.getElementById('f_inspDateGregorian');       // تاريخ الفحص (ميلادي)
  const fIssue= document.getElementById('f_inspectionIssueDate');     // تاريخ إصدار الفحص (اختياري)
  const fDoc  = document.getElementById('f_docSerialNum');            // رقم الوثيقة

  if (!fDoc) return;

  // علّم إن كانت القيمة الحالية مولّدة أو يدوية
  const initialDoc = (fDoc.value || '').trim();
  fDoc.dataset.autofill = looksAuto(initialDoc) ? '1' : (initialDoc ? '0' : '1');

  function looksAuto(val){
    // شكل تلقائي متوقع: YYYYMMDD-XXXXXX
    return /^[0-9]{8}-[0-9]{1,}$/.test(val);
  }

  function padLeft(num, size){
    num = String(num || '');
    while (num.length < size) num = '0' + num;
    return num;
  }

  function ymdFromDateInput(v){
    // v بشكل input[type=date] => YYYY-MM-DD
    if (!v) return '';
    const m = v.match(/^(\d{4})-(\d{2})-(\d{2})$/);
    if (!m) return '';
    return m[1] + m[2] + m[3]; // YYYYMMDD
  }

  function generateDocSerial(dateStr){
    const ymd = ymdFromDateInput(dateStr);
    if (!ymd) return '';
    return ymd + '-' + padLeft(idVal, 6);
  }

  function maybeAutofill(from){
    // لا نكتب فوق قيمة كتبها المستخدم يدويًا
    const canWrite = (fDoc.dataset.autofill === '1') || (fDoc.value.trim() === '');
    if (!canWrite) return;

    const dateSource = (from && from.value) ? from.value : (fDate && fDate.value ? fDate.value : '');
    if (!dateSource) return;

    const serial = generateDocSerial(dateSource);
    if (serial){
      fDoc.value = serial;
      fDoc.dataset.autofill = '1'; // لا يزال تلقائيًا
    }
  }

  // عند تغيير تاريخ الفحص — يولّد الرقم
  if (fDate)  fDate.addEventListener('change', () => maybeAutofill(fDate));
  // اختياريًا: لو تغيّر تاريخ الإصدار ولم يكن هناك تاريخ فحص
  if (fIssue) fIssue.addEventListener('change', () => {
    if (fDate && fDate.value) return; // نعتمد تاريخ الفحص أولًا
    maybeAutofill(fIssue);
  });

  // إذا الحقل فارغ عند التحميل وحقل التاريخ موجود—املأه الآن
  if (initialDoc === '' && fDate && fDate.value){
    maybeAutofill(fDate);
  }

  // إذا عدّل المستخدم docSerialNum يدويًا—أوقف الكتابة التلقائية لاحقًا
  fDoc.addEventListener('input', () => {
    const v = fDoc.value.trim();
    // إن غيّر لقيمة لا تطابق نمطنا، نفترض أنه يدوي
    fDoc.dataset.autofill = looksAuto(v) ? '1' : (v ? '0' : fDoc.dataset.autofill);
  });
})();
</script>
</body>
</html>
